//go:generate packer-sdc struct-markdown
//go:generate packer-sdc mapstructure-to-hcl2 -type ConnectConfig

package common

import (
	"context"
	"fmt"

	"github.com/hashicorp/packer-plugin-sdk/multistep"
	"github.com/hashicorp/packer-plugin-vsphere/builder/vsphere/driver"
)

type ConnectConfig struct {
	// vCenter Server hostname.
	VCenterServer string `mapstructure:"vcenter_server"`
	// vSphere username.
	Username string `mapstructure:"username"`
	// vSphere password.
	Password string `mapstructure:"password"`
	// Do not validate the vCenter Server TLS certificate. Defaults to `false`.
	InsecureConnection bool `mapstructure:"insecure_connection"`
	// vSphere datacenter name. Required if there is more than one datacenter in the vSphere inventory.
	Datacenter string `mapstructure:"datacenter"`
}

func (c *ConnectConfig) Prepare() []error {
	var errs []error

	if c.VCenterServer == "" {
		errs = append(errs, fmt.Errorf("'vcenter_server' is required"))
	}
	if c.Username == "" {
		errs = append(errs, fmt.Errorf("'username' is required"))
	}
	if c.Password == "" {
		errs = append(errs, fmt.Errorf("'password' is required"))
	}

	return errs
}

type StepConnect struct {
	Config *ConnectConfig
}

func (s *StepConnect) Run(_ context.Context, state multistep.StateBag) multistep.StepAction {
	d, err := driver.NewDriver(&driver.ConnectConfig{
		VCenterServer:      s.Config.VCenterServer,
		Username:           s.Config.Username,
		Password:           s.Config.Password,
		InsecureConnection: s.Config.InsecureConnection,
		Datacenter:         s.Config.Datacenter,
	})
	if err != nil {
		state.Put("error", err)
		return multistep.ActionHalt
	}
	state.Put("driver", d)

	return multistep.ActionContinue
}

func (s *StepConnect) Cleanup(multistep.StateBag) {}
