#![cfg(ed448)]

use wolfssl::wolfcrypt::random::RNG;
use wolfssl::wolfcrypt::ed448::*;

#[test]
#[cfg(all(ed448_import, ed448_export))]
fn test_make_public() {
    let mut rng = RNG::new().expect("Error creating RNG");
    let ed = Ed448::generate(&mut rng).expect("Error with generate()");
    let mut private = [0u8; Ed448::KEY_SIZE];
    ed.export_private_only(&mut private).expect("Error with export_private_only()");
    let mut ed = Ed448::new().expect("Error with new()");
    ed.import_private_only(&private).expect("Error with import_private_only()");
    let mut public = [0u8; Ed448::KEY_SIZE];
    ed.make_public(&mut public).expect("Error with make_public()");
}

#[test]
fn test_check_key() {
    let mut rng = RNG::new().expect("Error creating RNG");
    let mut ed = Ed448::generate(&mut rng).expect("Error with generate()");
    ed.check_key().expect("Error with check_key()");
}

#[test]
#[cfg(all(ed448_import, ed448_sign, ed448_verify))]
fn test_sign_verify() {
    let private_key = [
        0xc4u8, 0xea, 0xb0, 0x5d, 0x35, 0x70, 0x07, 0xc6,
        0x32, 0xf3, 0xdb, 0xb4, 0x84, 0x89, 0x92, 0x4d,
        0x55, 0x2b, 0x08, 0xfe, 0x0c, 0x35, 0x3a, 0x0d,
        0x4a, 0x1f, 0x00, 0xac, 0xda, 0x2c, 0x46, 0x3a,
        0xfb, 0xea, 0x67, 0xc5, 0xe8, 0xd2, 0x87, 0x7c,
        0x5e, 0x3b, 0xc3, 0x97, 0xa6, 0x59, 0x94, 0x9e,
        0xf8, 0x02, 0x1e, 0x95, 0x4e, 0x0a, 0x12, 0x27,
        0x4e
    ];
    let public_key = [
        0x43u8, 0xba, 0x28, 0xf4, 0x30, 0xcd, 0xff, 0x45,
        0x6a, 0xe5, 0x31, 0x54, 0x5f, 0x7e, 0xcd, 0x0a,
        0xc8, 0x34, 0xa5, 0x5d, 0x93, 0x58, 0xc0, 0x37,
        0x2b, 0xfa, 0x0c, 0x6c, 0x67, 0x98, 0xc0, 0x86,
        0x6a, 0xea, 0x01, 0xeb, 0x00, 0x74, 0x28, 0x02,
        0xb8, 0x43, 0x8e, 0xa4, 0xcb, 0x82, 0x16, 0x9c,
        0x23, 0x51, 0x60, 0x62, 0x7b, 0x4c, 0x3a, 0x94,
        0x80
    ];
    let message = [0x03u8];
    let context = [0x66u8,0x6f,0x6f];
    let expected_signature = [
        0xd4u8, 0xf8, 0xf6, 0x13, 0x17, 0x70, 0xdd, 0x46,
        0xf4, 0x08, 0x67, 0xd6, 0xfd, 0x5d, 0x50, 0x55,
        0xde, 0x43, 0x54, 0x1f, 0x8c, 0x5e, 0x35, 0xab,
        0xbc, 0xd0, 0x01, 0xb3, 0x2a, 0x89, 0xf7, 0xd2,
        0x15, 0x1f, 0x76, 0x47, 0xf1, 0x1d, 0x8c, 0xa2,
        0xae, 0x27, 0x9f, 0xb8, 0x42, 0xd6, 0x07, 0x21,
        0x7f, 0xce, 0x6e, 0x04, 0x2f, 0x68, 0x15, 0xea,
        0x00, 0x0c, 0x85, 0x74, 0x1d, 0xe5, 0xc8, 0xda,
        0x11, 0x44, 0xa6, 0xa1, 0xab, 0xa7, 0xf9, 0x6d,
        0xe4, 0x25, 0x05, 0xd7, 0xa7, 0x29, 0x85, 0x24,
        0xfd, 0xa5, 0x38, 0xfc, 0xcb, 0xbb, 0x75, 0x4f,
        0x57, 0x8c, 0x1c, 0xad, 0x10, 0xd5, 0x4d, 0x0d,
        0x54, 0x28, 0x40, 0x7e, 0x85, 0xdc, 0xbc, 0x98,
        0xa4, 0x91, 0x55, 0xc1, 0x37, 0x64, 0xe6, 0x6c,
        0x3c, 0x00
    ];

    let mut ed = Ed448::new().expect("Error with new()");
    ed.import_private_key(&private_key, Some(&public_key)).expect("Error with import_private_key()");

    let mut signature = [0u8; Ed448::SIG_SIZE];
    ed.sign_msg(&message, Some(&context), &mut signature).expect("Error with sign_msg()");
    assert_eq!(signature, expected_signature);

    let signature_valid = ed.verify_msg(&signature, &message, Some(&context)).expect("Error with verify_msg()");
    assert!(signature_valid);

    let mut signature = [0u8; Ed448::SIG_SIZE];
    ed.sign_msg_ex(&message, Some(&context), Ed448::ED448, &mut signature).expect("Error with sign_msg_ex()");
    assert_eq!(signature, expected_signature);

    let signature_valid = ed.verify_msg_ex(&signature, &message, Some(&context), Ed448::ED448).expect("Error with verify_msg_ex()");
    assert!(signature_valid);
}

#[test]
#[cfg(all(ed448_import, ed448_sign, ed448_streaming_verify))]
fn test_sign_streaming_verify() {
    let private_key = [
        0xc4u8, 0xea, 0xb0, 0x5d, 0x35, 0x70, 0x07, 0xc6,
        0x32, 0xf3, 0xdb, 0xb4, 0x84, 0x89, 0x92, 0x4d,
        0x55, 0x2b, 0x08, 0xfe, 0x0c, 0x35, 0x3a, 0x0d,
        0x4a, 0x1f, 0x00, 0xac, 0xda, 0x2c, 0x46, 0x3a,
        0xfb, 0xea, 0x67, 0xc5, 0xe8, 0xd2, 0x87, 0x7c,
        0x5e, 0x3b, 0xc3, 0x97, 0xa6, 0x59, 0x94, 0x9e,
        0xf8, 0x02, 0x1e, 0x95, 0x4e, 0x0a, 0x12, 0x27,
        0x4e
    ];
    let public_key = [
        0x43u8, 0xba, 0x28, 0xf4, 0x30, 0xcd, 0xff, 0x45,
        0x6a, 0xe5, 0x31, 0x54, 0x5f, 0x7e, 0xcd, 0x0a,
        0xc8, 0x34, 0xa5, 0x5d, 0x93, 0x58, 0xc0, 0x37,
        0x2b, 0xfa, 0x0c, 0x6c, 0x67, 0x98, 0xc0, 0x86,
        0x6a, 0xea, 0x01, 0xeb, 0x00, 0x74, 0x28, 0x02,
        0xb8, 0x43, 0x8e, 0xa4, 0xcb, 0x82, 0x16, 0x9c,
        0x23, 0x51, 0x60, 0x62, 0x7b, 0x4c, 0x3a, 0x94,
        0x80
    ];
    let message = [0x03u8];
    let context = [0x66u8,0x6f,0x6f];
    let expected_signature = [
        0xd4u8, 0xf8, 0xf6, 0x13, 0x17, 0x70, 0xdd, 0x46,
        0xf4, 0x08, 0x67, 0xd6, 0xfd, 0x5d, 0x50, 0x55,
        0xde, 0x43, 0x54, 0x1f, 0x8c, 0x5e, 0x35, 0xab,
        0xbc, 0xd0, 0x01, 0xb3, 0x2a, 0x89, 0xf7, 0xd2,
        0x15, 0x1f, 0x76, 0x47, 0xf1, 0x1d, 0x8c, 0xa2,
        0xae, 0x27, 0x9f, 0xb8, 0x42, 0xd6, 0x07, 0x21,
        0x7f, 0xce, 0x6e, 0x04, 0x2f, 0x68, 0x15, 0xea,
        0x00, 0x0c, 0x85, 0x74, 0x1d, 0xe5, 0xc8, 0xda,
        0x11, 0x44, 0xa6, 0xa1, 0xab, 0xa7, 0xf9, 0x6d,
        0xe4, 0x25, 0x05, 0xd7, 0xa7, 0x29, 0x85, 0x24,
        0xfd, 0xa5, 0x38, 0xfc, 0xcb, 0xbb, 0x75, 0x4f,
        0x57, 0x8c, 0x1c, 0xad, 0x10, 0xd5, 0x4d, 0x0d,
        0x54, 0x28, 0x40, 0x7e, 0x85, 0xdc, 0xbc, 0x98,
        0xa4, 0x91, 0x55, 0xc1, 0x37, 0x64, 0xe6, 0x6c,
        0x3c, 0x00
    ];

    let mut ed = Ed448::new().expect("Error with new()");
    ed.import_private_key(&private_key, Some(&public_key)).expect("Error with import_private_key()");

    let mut signature = [0u8; Ed448::SIG_SIZE];
    ed.sign_msg(&message, Some(&context), &mut signature).expect("Error with sign_msg()");
    assert_eq!(signature, expected_signature);

    ed.verify_msg_init(&signature, Some(&context), Ed448::ED448).expect("Error with verify_msg_init()");
    ed.verify_msg_update(&message).expect("Error with verify_msg_update()");
    let signature_valid = ed.verify_msg_final(&signature).expect("Error with verify_msg_final()");
    assert!(signature_valid);
}

#[test]
#[cfg(all(ed448_import, ed448_sign, ed448_verify))]
fn test_ph_sign_verify() {
    let private_key = [
        0x83u8, 0x3f, 0xe6, 0x24, 0x09, 0x23, 0x7b, 0x9d,
        0x62, 0xec, 0x77, 0x58, 0x75, 0x20, 0x91, 0x1e,
        0x9a, 0x75, 0x9c, 0xec, 0x1d, 0x19, 0x75, 0x5b,
        0x7d, 0xa9, 0x01, 0xb9, 0x6d, 0xca, 0x3d, 0x42,
        0xef, 0x78, 0x22, 0xe0, 0xd5, 0x10, 0x41, 0x27,
        0xdc, 0x05, 0xd6, 0xdb, 0xef, 0xde, 0x69, 0xe3,
        0xab, 0x2c, 0xec, 0x7c, 0x86, 0x7c, 0x6e, 0x2c,
        0x49
    ];
    let public_key = [
        0x25u8, 0x9b, 0x71, 0xc1, 0x9f, 0x83, 0xef, 0x77,
        0xa7, 0xab, 0xd2, 0x65, 0x24, 0xcb, 0xdb, 0x31,
        0x61, 0xb5, 0x90, 0xa4, 0x8f, 0x7d, 0x17, 0xde,
        0x3e, 0xe0, 0xba, 0x9c, 0x52, 0xbe, 0xb7, 0x43,
        0xc0, 0x94, 0x28, 0xa1, 0x31, 0xd6, 0xb1, 0xb5,
        0x73, 0x03, 0xd9, 0x0d, 0x81, 0x32, 0xc2, 0x76,
        0xd5, 0xed, 0x3d, 0x5d, 0x01, 0xc0, 0xf5, 0x38,
        0x80
    ];
    let message = [0x61u8,0x62,0x63];
    let context = [0x66u8,0x6f,0x6f];
    let hash = [
        0x48u8, 0x33, 0x66, 0x60, 0x13, 0x60, 0xa8, 0x77,
        0x1c, 0x68, 0x63, 0x08, 0x0c, 0xc4, 0x11, 0x4d,
        0x8d, 0xb4, 0x45, 0x30, 0xf8, 0xf1, 0xe1, 0xee,
        0x4f, 0x94, 0xea, 0x37, 0xe7, 0x8b, 0x57, 0x39,
        0xd5, 0xa1, 0x5b, 0xef, 0x18, 0x6a, 0x53, 0x86,
        0xc7, 0x57, 0x44, 0xc0, 0x52, 0x7e, 0x1f, 0xaa,
        0x9f, 0x87, 0x26, 0xe4, 0x62, 0xa1, 0x2a, 0x4f,
        0xeb, 0x06, 0xbd, 0x88, 0x01, 0xe7, 0x51, 0xe4
    ];
    let expected_signature = [
        0xc3u8, 0x22, 0x99, 0xd4, 0x6e, 0xc8, 0xff, 0x02,
        0xb5, 0x45, 0x40, 0x98, 0x28, 0x14, 0xdc, 0xe9,
        0xa0, 0x58, 0x12, 0xf8, 0x19, 0x62, 0xb6, 0x49,
        0xd5, 0x28, 0x09, 0x59, 0x16, 0xa2, 0xaa, 0x48,
        0x10, 0x65, 0xb1, 0x58, 0x04, 0x23, 0xef, 0x92,
        0x7e, 0xcf, 0x0a, 0xf5, 0x88, 0x8f, 0x90, 0xda,
        0x0f, 0x6a, 0x9a, 0x85, 0xad, 0x5d, 0xc3, 0xf2,
        0x80, 0xd9, 0x12, 0x24, 0xba, 0x99, 0x11, 0xa3,
        0x65, 0x3d, 0x00, 0xe4, 0x84, 0xe2, 0xce, 0x23,
        0x25, 0x21, 0x48, 0x1c, 0x86, 0x58, 0xdf, 0x30,
        0x4b, 0xb7, 0x74, 0x5a, 0x73, 0x51, 0x4c, 0xdb,
        0x9b, 0xf3, 0xe1, 0x57, 0x84, 0xab, 0x71, 0x28,
        0x4f, 0x8d, 0x07, 0x04, 0xa6, 0x08, 0xc5, 0x4a,
        0x6b, 0x62, 0xd9, 0x7b, 0xeb, 0x51, 0x1d, 0x13,
        0x21, 0x00
    ];

    let mut ed = Ed448::new().expect("Error with new()");
    ed.import_private_key(&private_key, Some(&public_key)).expect("Error with import_private_key()");

    let mut signature = [0u8; Ed448::SIG_SIZE];
    ed.sign_msg_ph(&message, Some(&context), &mut signature).expect("Error with sign_msg_ph()");
    assert_eq!(signature, expected_signature);

    let signature_valid = ed.verify_msg_ph(&signature, &message, Some(&context)).expect("Error with verify_msg_ph()");
    assert!(signature_valid);

    let mut signature = [0u8; Ed448::SIG_SIZE];
    ed.sign_hash_ph(&hash, Some(&context), &mut signature).expect("Error with sign_hash_ph()");
    assert_eq!(signature, expected_signature);

    let signature_valid = ed.verify_hash_ph(&signature, &hash, Some(&context)).expect("Error with verify_hash_ph()");
    assert!(signature_valid);
}

#[test]
#[cfg(all(ed448_import, ed448_export))]
fn test_import_export() {
    let mut rng = RNG::new().expect("Error creating RNG");
    let ed = Ed448::generate(&mut rng).expect("Error with generate()");

    let mut private = [0u8; Ed448::PRV_KEY_SIZE];
    let mut public = [0u8; Ed448::PUB_KEY_SIZE];
    ed.export_key(&mut private, &mut public).expect("Error with export_key()");

    let mut public2 = [0u8; Ed448::PUB_KEY_SIZE];
    ed.export_public(&mut public2).expect("Error with export_public()");
    assert_eq!(public2, public);

    let mut private2 = [0u8; Ed448::PRV_KEY_SIZE];
    ed.export_private(&mut private2).expect("Error with export_private()");
    assert_eq!(private2, private);

    let mut private_only = [0u8; Ed448::KEY_SIZE];
    ed.export_private_only(&mut private_only).expect("Error with export_private_only()");

    let mut ed = Ed448::new().expect("Error with new()");
    ed.import_private_key_ex(&private, Some(&public), false).expect("Error with import_private_key_ex()");

    let mut ed = Ed448::new().expect("Error with new()");
    ed.import_private_only(&private_only).expect("Error with import_private_only()");
    ed.import_public(&public).expect("Error with import_public()");
    ed.import_public_ex(&public, false).expect("Error with import_public_ex()");
}

#[test]
fn test_sizes() {
    let mut rng = RNG::new().expect("Error creating RNG");
    let ed = Ed448::generate(&mut rng).expect("Error with generate()");

    let size = ed.size().expect("Error with size()");
    assert_eq!(size, Ed448::KEY_SIZE);

    let size = ed.priv_size().expect("Error with priv_size()");
    assert_eq!(size, Ed448::PRV_KEY_SIZE);

    let size = ed.pub_size().expect("Error with pub_size()");
    assert_eq!(size, Ed448::PUB_KEY_SIZE);

    let size = ed.sig_size().expect("Error with sig_size()");
    assert_eq!(size, Ed448::SIG_SIZE);
}
